import io
import hashlib
import re
from datetime import datetime
from docx import Document
from docx.shared import Inches, Pt
from docx.enum.text import WD_ALIGN_PARAGRAPH
from docx.enum.style import WD_STYLE_TYPE
from services.NormalType import NormalService

class QuoteSvc(NormalService):

    def generageQuotetion(self, quotetionContent: str) -> str:
        """
        生成报价单文件，word文档，支持Markdown格式
        
        Args:
            quotetionContent (str): Markdown格式的报价单内容
            
        Returns:
            str: 上传后的文件信息，包含文件URL等信息
        """
        try:
            # 创建新的Word文档
            document = Document()
            
            # 设置文档样式
            self._setup_document_styles(document)
            
            # 解析Markdown内容并转换为Word格式
            self._parse_markdown_to_word(document, quotetionContent.value)
            
            # 将Document对象转换为二进制数据
            doc_buffer = io.BytesIO()
            document.save(doc_buffer)
            doc_binary_data = doc_buffer.getvalue()
            doc_buffer.close()
            
            # 生成文件名
            timestamp = datetime.now().strftime("%Y%m%d_%H%M%S")
            file_name = f"报价单_{timestamp}.docx"
            
            # 上传文件，设置过期时间为30天
            file_info = self.uploadByFile(
                file_name, 
                data=doc_binary_data, 
                contentType="application/vnd.openxmlformats-officedocument.wordprocessingml.document"
            )
            
            return [file_info]
            
        except Exception as e:
            error_msg = f"生成报价单时发生错误: {str(e)}"
            print(error_msg)
            return [{"error": error_msg}]
    
    def _setup_document_styles(self, document):
        """设置文档样式"""
        styles = document.styles
        
        # 设置标题样式
        for i in range(1, 7):
            try:
                heading_style = styles[f'Heading {i}']
            except KeyError:
                heading_style = styles.add_style(f'Heading {i}', WD_STYLE_TYPE.PARAGRAPH)
            
            if i == 1:
                heading_style.font.size = Pt(18)
                heading_style.font.bold = True
            elif i == 2:
                heading_style.font.size = Pt(16)
                heading_style.font.bold = True
            elif i == 3:
                heading_style.font.size = Pt(14)
                heading_style.font.bold = True
            else:
                heading_style.font.size = Pt(12)
                heading_style.font.bold = True
    
    def _parse_markdown_to_word(self, document, markdown_content):
        """解析Markdown内容并转换为Word格式"""
        lines = markdown_content.split('\n')
        i = 0
        
        while i < len(lines):
            line = lines[i].strip()
            
            if not line:
                i += 1
                continue
            
            # 处理标题
            if line.startswith('#'):
                level = len(line) - len(line.lstrip('#'))
                title_text = line.lstrip('#').strip()
                if title_text:
                    heading = document.add_heading(title_text, level=min(level, 6))
                    heading.style = f'Heading {min(level, 6)}'
            
            # 处理表格
            elif line.startswith('|') and '|' in line[1:]:
                table_data = self._parse_table(lines, i)
                if table_data:
                    self._create_table(document, table_data)
                    i += len(table_data) - 1
            
            # 处理列表
            elif line.startswith(('-', '*', '+')) or re.match(r'^\d+\.', line):
                list_items = self._parse_list(lines, i)
                self._create_list(document, list_items)
                i += len(list_items) - 1
            
            # 处理代码块
            elif line.startswith('```'):
                code_block = self._parse_code_block(lines, i)
                if code_block:
                    self._create_code_block(document, code_block)
                    i += len(code_block) + 1
            
            # 处理普通段落
            else:
                paragraph = document.add_paragraph()
                self._format_text(paragraph, line)
            
            i += 1
    
    def _parse_table(self, lines, start_index):
        """解析表格数据"""
        table_data = []
        i = start_index
        
        while i < len(lines):
            line = lines[i].strip()
            if not line.startswith('|'):
                break
            
            # 跳过分隔行
            if re.match(r'^\|[\s\-\|]+\|$', line):
                i += 1
                continue
            
            # 解析表格行
            cells = [cell.strip() for cell in line.split('|')[1:-1]]  # 去掉首尾空元素
            if cells:
                table_data.append(cells)
            
            i += 1
        
        return table_data
    
    def _create_table(self, document, table_data):
        """创建Word表格"""
        if not table_data:
            return
        
        table = document.add_table(rows=len(table_data), cols=len(table_data[0]))
        table.style = 'Table Grid'
        
        for i, row_data in enumerate(table_data):
            row = table.rows[i]
            for j, cell_data in enumerate(row_data):
                if j < len(row.cells):
                    cell = row.cells[j]
                    cell.text = cell_data
                    # 设置表格单元格样式
                    for paragraph in cell.paragraphs:
                        paragraph.alignment = WD_ALIGN_PARAGRAPH.CENTER
    
    def _parse_list(self, lines, start_index):
        """解析列表项"""
        list_items = []
        i = start_index
        
        while i < len(lines):
            line = lines[i].strip()
            if not line or (not line.startswith(('-', '*', '+')) and not re.match(r'^\d+\.', line)):
                break
            
            list_items.append(line)
            i += 1
        
        return list_items
    
    def _create_list(self, document, list_items):
        """创建Word列表"""
        for item in list_items:
            # 移除列表标记
            if re.match(r'^\d+\.', item):
                text = re.sub(r'^\d+\.\s*', '', item)
                document.add_paragraph(text, style='List Number')
            else:
                text = re.sub(r'^[-*+]\s*', '', item)
                document.add_paragraph(text, style='List Bullet')
    
    def _parse_code_block(self, lines, start_index):
        """解析代码块"""
        code_lines = []
        i = start_index + 1
        
        while i < len(lines):
            line = lines[i]
            if line.strip().startswith('```'):
                break
            code_lines.append(line)
            i += 1
        
        return code_lines
    
    def _create_code_block(self, document, code_lines):
        """创建代码块"""
        code_text = '\n'.join(code_lines)
        paragraph = document.add_paragraph()
        run = paragraph.add_run(code_text)
        run.font.name = 'Courier New'
        run.font.size = Pt(10)
        
        # 设置代码块背景色（灰色）
        paragraph.paragraph_format.left_indent = Inches(0.5)
        paragraph.paragraph_format.right_indent = Inches(0.5)
    
    def _format_text(self, paragraph, text):
        """格式化文本，处理粗体、斜体等"""
        # 处理粗体 **text** 或 __text__
        text = re.sub(r'\*\*(.*?)\*\*', r'**\1**', text)
        text = re.sub(r'__(.*?)__', r'**\1**', text)
        
        # 处理斜体 *text* 或 _text_
        text = re.sub(r'\*(.*?)\*', r'*\1*', text)
        text = re.sub(r'_(.*?)_', r'*\1*', text)
        
        # 分割文本并应用格式
        parts = re.split(r'(\*\*.*?\*\*|\*.*?\*)', text)
        
        for part in parts:
            if part.startswith('**') and part.endswith('**'):
                run = paragraph.add_run(part[2:-2])
                run.bold = True
            elif part.startswith('*') and part.endswith('*'):
                run = paragraph.add_run(part[1:-1])
                run.italic = True
            else:
                paragraph.add_run(part)

    def uploadByFile(
            self, fileName, data=None, contentType="application/octet-stream"
        ):
        # 上传文件
        # app是全局变量，无需导入
        storage = app.getElement("storages.Default")
        md5 = hashlib.md5(data).hexdigest()
        resp = storage.uploadByFile(md5, data, contentType)
        row = {
            "uid": md5,
            "name": md5,
            "storeFullName": "storages.Default",
            "uploadTime": datetime.now().strftime("%Y-%m-%d %H:%M:%S"),
            "md5": md5,
            "fileName": fileName,
            "url": resp["url"],
        }
        return row
